//------------------------------------------------------------------------------
// File: client_slgameobj.cs
// This file contains the client methods for the game objects
// Author: Matthew Rudge
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Tests if the game object is disabled for any reason
//! \retval bool True if disabled, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::isDisabled(%this)
{
   // If healing then we are disabled
   if(%this.isHealing) {
      return true;
   }
   
   // Check disabled character
   if(slgIsCharacter(%this)) {
      %cmpChar = slgQueryInterface(%this, $CID_CHARACTER);
      return %cmpChar.isDisabled();
   }
   
   // Check disabled building   
   if(slgIsBuilding(%this)) {
      %cmpBldg = slgQueryInterface(%this, $CID_BUILDING);
      return %cmpBldg.isDisabled();
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Called when an object's team number has been changed on the client
//! \param %oldTeam Old team number
//! \param %newTeam New team number
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::onTeamChange(%object, %oldTeam, %newTeam)
{
   if(%oldTeam == $OST_PLAYER || %newTeam == $OST_PLAYER) {
      // If this object is selected then repopulate hud
      if(isSelected(%object)) {
         slgPopulateCmdBtns();
      }
      
      // Update mini-map hud locator buttons
      slgUpdateLocatorButtons();
   }
   
   // choose new ambient sounds if necessary
   chooseAmbientSound();
   
   // check if this is a character
   %component = slgQueryInterface(%object, $CID_CHARACTER);
   if (isObject(%component) == false)
   {
      return;
   }
   
   // if the character object is not selected, we are done
   if (%object.selectron == 0)
   {
      return;
   }
   
   // remove the current selection on the character
   %object.selectron.stopSelectron();
   %object.selectron = 0;
   
   // find what the new selectron should be
   %selectron = 0;
   %team = %object.getTeam();
   if (%team == $OST_PLAYER)
   {
     %selectron = startSelectron(%object, $SELECT_PLAYER_EFFECT_ID);
   }
   else if (%team == $OST_NEUTRAL)
   {
     %selectron = startSelectron(%object, $SELECT_NEUTRAL_EFFECT_ID);
   }
   else
   {
     %selectron = startSelectron(%object, $SELECT_ENEMY_EFFECT_ID);
   }

   // add the selectron to the character
   if (%selectron != 0)
   {
     %selectron.addConstraint(%object, "selected");
     %object.selectron = %selectron; // add a field to handle this
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Called when an object is selected on the client
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::onClientSelect(%this)
{
   // turn on the selection arrow
   if(%this.selectIcon != 0) {
     %this.selectIcon.setSelectedImage();
   }
     
   // Play selection sound
   playSelectionSound(%this);
   
   if(slgIsCharacter(%this)) {
      // Create health bar
      CreateHealthBar(%this);
      
      // Add selectron based on team
      %selection = 0;
      %team = %this.getTeam();
      if(%team == $OST_PLAYER) {
        %selectron = startSelectron(%this, $SELECT_PLAYER_EFFECT_ID);
      }
      else if(%team == $OST_NEUTRAL) {
        %selectron = startSelectron(%this, $SELECT_NEUTRAL_EFFECT_ID);
      }
      else {
        %selectron = startSelectron(%this, $SELECT_ENEMY_EFFECT_ID);
      }
      if(%selectron != 0){
        %selectron.addConstraint(%this, "selected");
        %this.selectron = %selectron; // add a field to handle this
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Called when an object is unselected on the client
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::onClientUnselect(%this)
{
   // turn off the selection arrow
   if(%this.selectIcon != 0) {
     %this.selectIcon.setUpImage();
   }

   // remove the health bar
   DestroyHealthBar(%this);

   if(slgIsCharacter(%this)) {
      // turn off the selection decal
      if(%this.selectron) {
        %this.selectron.stopSelectron();
        %this.selectron = 0;
      }
      
      // Stop construnction sound if character is building
      %cmpChar = slgQueryInterface(%this, $CID_CHARACTER);
      if(%cmpChar.inState($CharacterState::Build)) {
         stopConstructionSound();
      }
   }
   else if(slgIsBuilding(%this)) {
      // Stop construction sound if building is under construction
      %cmpBldg = slgQueryInterface(%this, $CID_BUILDING);
      if(%cmpBldg.getState() == $BuildingState::Construction) {
         stopConstructionSound();
      }
   }
}

// End client_slgameobj.cs